"use client";

import { useEffect, useState } from "react";
import { useSearchParams } from "next/navigation";
import ProductCard from "../components/category/ProductCard";

export default function SearchPage() {
  const searchParams = useSearchParams();

  const searchKey = searchParams.get("search_key") || "";
  const categoryId = searchParams.get("category_id") || "";

  const [products, setProducts] = useState([]);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    if (!searchKey) {
      setProducts([]);
      return;
    }

    setLoading(true);

    const params = new URLSearchParams();
    params.append("search_key", searchKey);

    if (categoryId) {
      params.append("category_id", categoryId);
    }

    fetch(`${process.env.NEXT_PUBLIC_API_URL}/search?${params.toString()}`)
      .then((res) => res.json())
      .then((res) => {
        const apiProducts = res?.data?.products?.data ?? [];

        const formatted = apiProducts.map((item) => {
          const price = Number(item.base_price ?? 0);

          return {
            id: item.id,
            name: item.name,
            slug: item.slug,
            category: item.categories?.[0]?.name ?? "",
            originalPrice: price,
            salePrice: price,
            rating: item.reviews?.length ?? 0,
            image: item.main_image
              ? `${process.env.NEXT_PUBLIC_IMAGE_BASE}/product/${item.main_image}`
              : "/placeholder.png",
            badge: item.is_featured ? "FEATURED" : null,
            buttonType: item.has_variants ? "options" : "cart",
          };
        });

        setProducts(formatted);
      })
      .catch(() => {
        setProducts([]);
      })
      .finally(() => {
        setLoading(false);
      });
  }, [searchKey, categoryId]);

  return (
    <div className="max-w-7xl mx-auto px-4 py-8">
      <h1 className="text-xl font-semibold mb-6">
        Search Result for "{searchKey}"
      </h1>

      {loading && <p>Loading...</p>}

      {!loading && products.length === 0 && (
        <p>No product found</p>
      )}

      <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
        {products.map((item) => (
          <ProductCard key={item.id} product={item} />
        ))}
      </div>
    </div>
  );
}
