'use client';

import { useEffect, useState, useMemo } from 'react';
import { useSearchParams } from 'next/navigation';
import {
  Star,
  ChevronLeft,
  ChevronRight,
  Heart,
  ShoppingCart,
  ZoomIn,
} from 'lucide-react';
import Link from 'next/link';
import toast from 'react-hot-toast';
import Cookies from 'js-cookie';

const IMAGE_BASE = process.env.NEXT_PUBLIC_IMAGE_BASE + '/product';

export default function ProductPage() {
  const searchParams = useSearchParams();
  const slug = searchParams.get('slug');

  const [data, setData] = useState(null);
  const [loading, setLoading] = useState(true);

  const [quantity, setQuantity] = useState(1);
  const [isZoomed, setIsZoomed] = useState(false);
  const [mainImageIndex, setMainImageIndex] = useState(0);

  const [selectedColorAttrId, setSelectedColorAttrId] = useState(null);
  const [selectedSizeAttrId, setSelectedSizeAttrId] = useState(null);
  const [activeTab, setActiveTab] = useState('description');

  /* -----------------------------
     CART ID
  ------------------------------*/
  let cartId = Cookies.get('cart_id');
  if (!cartId) {
    cartId = crypto.randomUUID();
    Cookies.set('cart_id', cartId, { expires: 30 });
  }

  /* -----------------------------
     FETCH PRODUCT
  ------------------------------*/
  useEffect(() => {
    if (!slug) return;

    setLoading(true);
    fetch(`${process.env.NEXT_PUBLIC_API_URL}/product/${slug}`)
      .then(res => res.json())
      .then(res => {
        setData(res);
        setSelectedColorAttrId(null);
        setSelectedSizeAttrId(null);
        setMainImageIndex(0);
        setLoading(false);
      })
      .catch(() => setLoading(false));
  }, [slug]);

  /* -----------------------------
     STOCK CALCULATION
  ------------------------------*/
  const stockQty = useMemo(() => {
    if (!data || data.product.track_inventory !== 1) return null;

    const stocks = data.product.stocks || [];

    if (!selectedColorAttrId || !selectedSizeAttrId) {
      return stocks.reduce((sum, s) => sum + Number(s.quantity || 0), 0);
    }

    const selectedAttrs = [
      Number(selectedColorAttrId),
      Number(selectedSizeAttrId),
    ].sort((a, b) => a - b);

    const matched = stocks.find(s => {
      const stockAttrs = JSON.parse(s.attributes)
        .map(Number)
        .sort((a, b) => a - b);

      return JSON.stringify(stockAttrs) === JSON.stringify(selectedAttrs);
    });

    return matched ? Number(matched.quantity) : 0;
  }, [data, selectedColorAttrId, selectedSizeAttrId]);

  /* -----------------------------
     GUARDS
  ------------------------------*/
  if (!slug) return <p className="p-10">Invalid product</p>;
  if (loading) return <p className="p-10">Loading...</p>;
  if (!data) return <p>Product not found</p>;

  const { product, attributes, relatedProducts } = data;

  /* -----------------------------
     IMAGES
  ------------------------------*/
  const defaultImages = [
    product.main_image,
    ...(product.product_preview_images || []).map(i => i.image),
    ...(product.product_variant_images || []).map(i => i.image),
  ].filter(Boolean);

  const colorImages = selectedColorAttrId
    ? product.product_variant_images
      .filter(img => img.assign_product_attribute_id === selectedColorAttrId)
      .map(i => i.image)
    : [];

  const allImages =
    selectedColorAttrId && colorImages.length > 0
      ? colorImages
      : defaultImages;

  /* -----------------------------
     HELPERS
  ------------------------------*/
  const getAttrValues = attrId =>
    product.assign_attributes.filter(
      a => a.product_attribute_id === attrId
    );

  const getExtraPriceByAttrId = attrId => {
    if (!attrId) return 0;
    const attr = product.assign_attributes.find(a => a.id === attrId);
    return attr ? Number(attr.extra_price || 0) : 0;
  };

  const finalPrice =
    Number(product.base_price) +
    getExtraPriceByAttrId(selectedColorAttrId) +
    getExtraPriceByAttrId(selectedSizeAttrId);

  /* -----------------------------
     THUMB SCROLL
  ------------------------------*/
  const scrollThumbnails = dir => {
    const el = document.getElementById('thumbnail-container');
    if (!el) return;
    el.scrollBy({
      left: dir === 'left' ? -200 : 200,
      behavior: 'smooth',
    });
  };

  /* -----------------------------
     ADD TO CART
  ------------------------------*/
  const handleAddToCart = async () => {
    const toastId = toast.loading('Adding to cart...');

    try {
      const formData = new FormData();
      formData.append('product_id', product.id);
      formData.append('quantity', quantity);
      formData.append('cart_id', cartId);

      const attrs = [];
      if (selectedColorAttrId) attrs.push(Number(selectedColorAttrId));
      if (selectedSizeAttrId) attrs.push(Number(selectedSizeAttrId));
      attrs.sort((a, b) => a - b);
      attrs.forEach(id => formData.append('attributes[]', id));

      const res = await fetch(
        `${process.env.NEXT_PUBLIC_API_URL}/add-to-cart`,
        {
          method: 'POST',
          credentials: 'include',
          body: formData,
        }
      );

      const result = await res.json();

      if (result.success) {
        toast.success(result.success, { id: toastId });
        if (typeof result.count === "number") {
          Cookies.set('cart_count', 9, {
            expires: 30,
            path: '/',        // 🔥 MUST
          });
          // live update trigger
        }
      }
      else toast.error(result.error || 'Select product variants', { id: toastId });
    } catch {
      toast.error('Something went wrong', { id: toastId });
    }
  };

  /* =============================
     RENDER
  ==============================*/
  return (
    <div className="max-w-7xl mx-auto p-6">

      {/*  ================= IMAGES =================  */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
        <div>
          <div
            className="
        relative overflow-hidden rounded-xl bg-gray-50
        border border-gray-200
        cursor-zoom-in
        flex items-center justify-center
      "
            onMouseEnter={() => setIsZoomed(true)}
            onMouseLeave={() => setIsZoomed(false)}
          >
            <img
              src={`${IMAGE_BASE}/${allImages[mainImageIndex]}`}
              alt={product.name}
              className={`
          w-full h-auto object-contain
          transition-transform duration-300
          ${isZoomed ? 'scale-150' : 'scale-100'}
        `}
            />
            {isZoomed && (
              <div className="absolute top-3 right-3 bg-white p-1.5 rounded-full shadow">
                <ZoomIn className="w-5 h-5 text-gray-600" />
              </div>
            )}
          </div>

          {allImages.length > 1 && (
            <div className="mt-5 relative">
              <button
                onClick={() => scrollThumbnails('left')}
                className="
            absolute left-0 top-1/2 -translate-y-1/2
            bg-white p-2 shadow rounded-full z-10
            hover:bg-gray-100
          "
              >
                <ChevronLeft size={18} />
              </button>

              <button
                onClick={() => scrollThumbnails('right')}
                className="
            absolute right-0 top-1/2 -translate-y-1/2
            bg-white p-2 shadow rounded-full z-10
            hover:bg-gray-100
          "
              >
                <ChevronRight size={18} />
              </button>

              <div
                id="thumbnail-container"
                className="flex gap-3 overflow-x-auto px-12 scrollbar-hide"
              >
                {allImages.map((img, i) => (
                  <button
                    key={`${img}-${i}`}
                    onClick={() => setMainImageIndex(i)}
                    className={`
                w-24 h-24 flex-shrink-0 rounded-lg overflow-hidden
                border-2 transition-all
                ${mainImageIndex === i
                        ? 'border-black'
                        : 'border-gray-300 hover:border-gray-500'}
              `}
                  >
                    <img
                      src={`${IMAGE_BASE}/${img}`}
                      className="w-full h-full object-cover"
                    />
                  </button>
                ))}
              </div>
            </div>
          )}
        </div>

        {/* ================= INFO ================= */}
        <div>
          <h1 className="text-3xl font-semibold text-gray-800 mb-2">
            {product.name}
          </h1>

          {product.track_inventory === 1 && (
            <div className="mb-4">
              {stockQty > 0 ? (
                <span className="bg-green-100 text-green-700 px-3 py-1 rounded-full text-xs">
                  In Stock • {stockQty}
                </span>
              ) : (
                <span className="bg-red-100 text-red-700 px-3 py-1 rounded-full text-xs">
                  Out of Stock
                </span>
              )}
            </div>
          )}

          <div className="text-3xl font-bold text-gray-900 mb-6">
            ৳ {finalPrice.toFixed(2)}
          </div>

          {/* ATTRIBUTES */}
          {attributes.map(attr => {
            const values = getAttrValues(attr.product_attribute_id);

            if (attr.product_attribute.type === 2) {
              return (
                <div key={attr.product_attribute_id} className="mb-5">
                  <p className="font-medium text-sm mb-2">
                    {attr.product_attribute.name_for_user}
                  </p>
                  <div className="flex gap-3">
                    {values.map(v => (
                      <button
                        key={v.id}
                        onClick={() => setSelectedColorAttrId(v.id)}
                        className={`
                    w-9 h-9 rounded-full border-2
                    transition-all
                    ${selectedColorAttrId === v.id
                            ? 'border-black ring-2 ring-black'
                            : 'border-gray-300 hover:scale-110'}
                  `}
                        style={{ backgroundColor: `#${v.value}` }}
                      />
                    ))}
                  </div>
                </div>
              );
            }

            if (attr.product_attribute.type === 1) {
              return (
                <div key={attr.product_attribute_id} className="mb-5">
                  <p className="font-medium text-sm mb-2">
                    {attr.product_attribute.name_for_user}
                  </p>
                  <div className="flex gap-2">
                    {values.map(v => (
                      <button
                        key={v.id}
                        onClick={() => setSelectedSizeAttrId(v.id)}
                        className={`
                    px-4 py-2 text-sm rounded border transition-all
                    ${selectedSizeAttrId === v.id
                            ? 'bg-black text-white border-black'
                            : 'border-gray-300 hover:border-gray-500'}
                  `}
                      >
                        {v.value}
                      </button>
                    ))}
                  </div>
                </div>
              );
            }

            return null;
          })}

          {/* QTY + CART */}
          <div className="flex gap-4 mt-8">
            <div className="flex items-center border rounded-lg overflow-hidden">
              <button
                onClick={() => setQuantity(q => Math.max(1, q - 1))}
                className="px-4 py-2 hover:bg-gray-100"
              >
                −
              </button>
              <span className="px-5">{quantity}</span>
              <button
                onClick={() => setQuantity(q => q + 1)}
                className="px-4 py-2 hover:bg-gray-100"
              >
                +
              </button>
            </div>

            <button
              onClick={handleAddToCart}
              className="
          flex-1 bg-black text-white py-3 rounded-lg
          flex items-center justify-center gap-2
          hover:bg-gray-900 transition
        "
            >
              <ShoppingCart size={18} />
              Add to Cart
            </button>
          </div>
        </div>
      </div>






      {/* ================= PRODUCT TABS ================= */}
      <div className="mt-16">

        {/* TAB BUTTONS */}
        <div className="border-b border-gray-200">
          <div className="flex flex-wrap">
            <button
              onClick={() => setActiveTab('description')}
              className={`px-5 py-3 text-sm font-medium border-r
          ${activeTab === 'description'
                  ? 'bg-blue-600 text-white'
                  : 'bg-gray-100 text-gray-700 hover:bg-gray-200'}
        `}
            >
              Description
            </button>

            <button
              onClick={() => setActiveTab('specification')}
              className={`px-5 py-3 text-sm font-medium border-r
          ${activeTab === 'specification'
                  ? 'bg-blue-600 text-white'
                  : 'bg-gray-100 text-gray-700 hover:bg-gray-200'}
        `}
            >
              Specification
            </button>

            <button
              onClick={() => setActiveTab('video')}
              className={`px-5 py-3 text-sm font-medium border-r
          ${activeTab === 'video'
                  ? 'bg-blue-600 text-white'
                  : 'bg-gray-100 text-gray-700 hover:bg-gray-200'}
        `}
            >
              Video
            </button>

            <button
              onClick={() => setActiveTab('reviews')}
              className={`px-5 py-3 text-sm font-medium
          ${activeTab === 'reviews'
                  ? 'bg-blue-600 text-white'
                  : 'bg-gray-100 text-gray-700 hover:bg-gray-200'}
        `}
            >
              Reviews ({product.reviews?.length || 0})
            </button>
          </div>
        </div>

        {/* TAB CONTENT */}
        <div className="bg-white p-6 border border-t-0">

          {/* DESCRIPTION */}
          {activeTab === 'description' && (
            <div className="text-gray-700 leading-relaxed">
              {product.description ? (
                <div dangerouslySetInnerHTML={{ __html: product.description }} />
              ) : (
                <p>{data.seoContents?.description || 'No description available.'}</p>
              )}
            </div>
          )}

          {/* SPECIFICATION */}
          {activeTab === 'specification' && (
            <div className="text-gray-700 leading-relaxed">
              {product.specification ? (
                <div dangerouslySetInnerHTML={{ __html: product.specification }} />
              ) : (
                <p>No specification available.</p>
              )}
            </div>
          )}

          {/* VIDEO */}
          {activeTab === 'video' && (
            <div className="max-w-xl">
              {product.video_link ? (
                <div className="aspect-video">
                  <iframe
                    src={product.video_link}
                    className="w-full h-full rounded"
                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                    allowFullScreen
                  />
                </div>
              ) : (
                <p className="text-gray-500">No video available.</p>
              )}
            </div>
          )}

          {/* REVIEWS */}
          {activeTab === 'reviews' && (
            <div>
              {product.reviews && product.reviews.length > 0 ? (
                <div className="space-y-4">
                  {product.reviews.map((review, i) => (
                    <div key={i} className="border-b pb-3">
                      <p className="font-medium">{review.user_name || 'User'}</p>
                      <p className="text-sm text-gray-600">{review.comment}</p>
                    </div>
                  ))}
                </div>
              ) : (
                <p className="text-gray-500">No reviews yet.</p>
              )}
            </div>
          )}

        </div>
      </div>




      {/* ================= RELATED PRODUCTS ================= */}
      {relatedProducts && relatedProducts.length > 0 && (
        <div className="mt-14">
          <h3 className="text-xl font-bold mb-6">Related Products</h3>

          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            {relatedProducts.slice(0, 4).map((p, index) => (
              <div
                key={`${p.id}-${index}`}
                className="
            group relative bg-white border border-gray-100 overflow-hidden
            transition-shadow duration-300 hover:shadow-md
          "
              >
                {/* IMAGE */}
                <Link href={`/product-details?slug=${p.slug}`} className="block">
                  <div className="relative overflow-hidden">
                    <img
                      src={`${IMAGE_BASE}/${p.main_image}`}
                      alt={p.name}
                      className="
                  w-full h-64 object-cover
                  transition-transform duration-300
                  group-hover:scale-105
                "
                    />

                    {/* BADGE (optional) */}
                    {(p.discount || p.is_hot) && (
                      <div className="absolute top-3 left-3 flex gap-2">
                        {p.is_hot && (
                          <span className="bg-green-500 text-white text-xs px-2 py-1">
                            HOT
                          </span>
                        )}
                        {p.discount && (
                          <span className="bg-red-500 text-white text-xs px-2 py-1">
                            {p.discount}%
                          </span>
                        )}
                      </div>
                    )}
                  </div>
                </Link>

                {/* INFO */}
                <div className="p-4 text-center">
                  {/* Category (optional placeholder) */}
                  <p className="text-xs text-gray-400 uppercase">
                    {p.category_name || "Product"}
                  </p>

                  {/* TITLE */}
                  <h3 className="text-sm font-medium mt-1 text-gray-600 line-clamp-2">
                    {p.name}
                  </h3>

                  {/* RATING (static for now) */}
                  <div className="flex justify-center text-yellow-400 text-xs mt-1">
                    ★★★★★
                  </div>

                  {/* PRICE */}
                  <div className="mt-2">
                    {p.old_price && p.old_price > p.base_price && (
                      <span className="text-sm text-gray-400 line-through mr-2">
                        ৳ {Number(p.old_price).toFixed(2)}
                      </span>
                    )}

                    <span className="text-lg font-semibold text-gray-800">
                      ৳ {Number(p.base_price).toFixed(2)}
                    </span>
                  </div>

                  {/* ACTION ROW */}
                  <div className="mt-3 flex items-center justify-center gap-2">
                    {/* WISHLIST */}
                    <button
                      className="
                  border border-gray-400 p-2 text-gray-400
                  opacity-0 group-hover:opacity-100
                  transition-opacity hover:bg-gray-100
                "
                    >
                      <Heart size={16} />
                    </button>

                    {/* ADD TO CART */}
                    <button
                      className="
                  flex items-center justify-center gap-2 px-4 py-2
                  border border-gray-400 text-gray-400
                  text-sm font-medium
                  transition-colors
                  group-hover:bg-black group-hover:text-white group-hover:border-black
                "
                    >
                      <ShoppingCart size={16} />
                      Add to Cart
                    </button>

                    {/* VIEW */}
                    <Link
                      href={`/product-details?slug=${p.slug}`}
                      className="
                  border border-gray-400 p-2 text-gray-400
                  opacity-0 group-hover:opacity-100
                  transition-opacity hover:bg-gray-100
                "
                    >
                      <ZoomIn size={16} />
                    </Link>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}


    </div>
  );
}
