import { Heart, ShoppingCart, Eye } from "lucide-react";
import Link from "next/link";
import { useState } from "react";

const FeaturedProductCard = ({
  id,
  main_image,
  category,
  name,
  base_price,
  rating,
  isHot,
  discount,
  hasCountdown,
  hasQuickView,
  slug,
}) => {
  const [isWishlisted, setIsWishlisted] = useState(false);


  const imageUrl = main_image
    ? `${process.env.NEXT_PUBLIC_IMAGE_BASE}/product/${main_image}`
    : "/placeholder.png";

  // Title character limit
  const truncate = (text, limit = 40) => {
    if (!text) return "";
    return text.length > limit ? text.slice(0, limit) + "..." : text;
  };

  return (
    <div className="bg-white border border-gray-200 overflow-hidden group hover:shadow-lg transition-shadow">
      {/* Image Container */}
      <div className="relative overflow-hidden bg-ecom-light-gray aspect-square">
        <Link href={`/product-details?slug=${slug}`} className="block">
          <img
            src={imageUrl}
            alt={name}
            className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
          />
        </Link>
        {/* Badges */}
        <div className="absolute top-3 left-3 flex flex-col gap-2">
          {isHot && (
            <span className="bg-green-500 text-white px-3 py-1 rounded text-xs font-semibold">
              HOT
            </span>
          )}
          {discount && (
            <span className="bg-ecom-red text-white px-3 py-1 rounded text-xs font-semibold">
              -{discount}%
            </span>
          )}
        </div>

        {/* Cart Button (NEW) */}

        {!hasCountdown && !hasQuickView && (
          <button
            className="absolute top-3 right-3 bg-blue-500 text-white p-2 rounded-full shadow-md 
             hover:bg-ecom-blue hover:text-white transition-colors
             opacity-0 group-hover:opacity-100"
            aria-label="Add to cart"
          >
            <ShoppingCart className="w-5 h-5" />
          </button>
        )}


        {/* Action Buttons */}

        <button className="absolute bottom-0 left-0 right-0 bg-blue-400 text-white py-3 text-sm font-semibold opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center gap-2">
          <Eye className="w-4 h-4" />
          QUICK VIEW
        </button>



      </div>

      {/* Product Info */}
      <div className="p-4">
        <p className="text-xs text-ecom-gray uppercase mb-2">{category}</p>


        {/* Title + Wishlist */}
        <div className="flex items-start justify-between gap-2 mb-2">
          <h3 className="font-semibold text-base hover:text-ecom-blue cursor-pointer transition-colors leading-snug">
            {truncate(name, 20)}
          </h3>

          <button
            onClick={() => setIsWishlisted(!isWishlisted)}
            className="text-gray-400 hover:text-ecom-red transition-colors flex-shrink-0"
            aria-label="Add to wishlist"
          >
            <Heart
              className={`w-5 h-5 ${isWishlisted ? "fill-current text-ecom-red" : ""
                }`}
            />
          </button>
        </div>


        {/* Rating */}
        <div className="flex items-center gap-1 mb-3">
          {Array.from({ length: 5 }).map((_, index) => (
            <svg
              key={index}
              className={`w-4 h-4 ${index < rating
                ? "text-yellow-400 fill-current"
                : "text-gray-300"
                }`}
              viewBox="0 0 20 20"
            >
              <path d="M10 15l-5.878 3.09 1.123-6.545L.489 6.91l6.572-.955L10 0l2.939 5.955 6.572.955-4.756 4.635 1.123 6.545z" />
            </svg>
          ))}
        </div>

        {/* base_price */}
        <div className="text-xl font-bold text-ecom-dark">
          ৳{Number(base_price).toFixed(2)}

        </div>
      </div>

    </div>
  );
};

export default FeaturedProductCard;
