"use client";

import { useEffect, useState } from "react";
import Link from "next/link";
import { ChevronDown, ChevronUp, Minus } from "lucide-react";

const Sidebar = ({
  activeCategoryId,
  categories = [],   // ✅ default array
  brands = [],       // ✅ default array
  selectedBrands = [],
  priceRange = [0, 1000],
  onBrandChange,
  onPriceChange,
}) => {
  const [expandedCategories, setExpandedCategories] = useState([]);

  const minPrice = 0;
  const maxPrice = 1000;

  /* -----------------------------
     TOGGLE BRAND
  ------------------------------*/
  const toggleBrand = (id) => {
    if (selectedBrands.includes(id)) {
      onBrandChange(selectedBrands.filter((b) => b !== id));
    } else {
      onBrandChange([...selectedBrands, id]);
    }
  };

  /* -----------------------------
     AUTO OPEN CATEGORY
  ------------------------------*/
  const findCategoryPath = (items, targetId, path = []) => {
    for (const item of items) {
      const id = String(item.id);
      const newPath = [...path, id];

      if (id === String(targetId)) return newPath;

      if (item.all_subcategories?.length) {
        const found = findCategoryPath(
          item.all_subcategories,
          targetId,
          newPath
        );
        if (found) return found;
      }
    }
    return null;
  };

  useEffect(() => {
    if (!categories?.length || !activeCategoryId) return;
    const path = findCategoryPath(categories, activeCategoryId);
    if (path) setExpandedCategories(path);
  }, [categories, activeCategoryId]);

  /* -----------------------------
     RENDER CATEGORY TREE
  ------------------------------*/
const renderCategories = (items = [], level = 0) => {
  if (!Array.isArray(items) || items.length === 0) return null;

  return (
    <ul className={level === 0 ? "space-y-2" : "ml-4 mt-2 space-y-1"}>
      {items.map((cat) => {
        const id = String(cat.id);
        const isOpen = expandedCategories.includes(id);
        const isActive = id === String(activeCategoryId);
        const hasChildren = cat.all_subcategories?.length > 0;

        return (
          <li key={id}>
            <div className="flex items-center justify-between">
              <Link
                href={`/category/${id}`}
                className={`text-sm transition-colors ${
                  isActive
                    ? "text-black font-medium"
                    : "text-gray-500 hover:text-gray-700"
                }`}
              >
                {cat.name}
              </Link>

              {hasChildren && (
                <button
                  onClick={() =>
                    setExpandedCategories((prev) =>
                      prev.includes(id)
                        ? prev.filter((x) => x !== id)
                        : [...prev, id]
                    )
                  }
                  className="text-gray-400 hover:text-gray-600"
                >
                  {isOpen ? <ChevronUp size={14} /> : <ChevronDown size={14} />}
                </button>
              )}
            </div>

            {isOpen && hasChildren &&
              renderCategories(cat.all_subcategories, level + 1)}
          </li>
        );
      })}
    </ul>
  );
};


  return (
    <aside className="w-full lg:w-72 border border-gray-200 p-4">
      {/* Categories */}
      <div className="mb-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-sm font-semibold uppercase text-gray-700">
            Categories
          </h3>
          <Minus size={14} className="text-gray-400" />
        </div>

        {renderCategories(categories)}
      </div>

      {/* Price Filter */}
      <div className="mb-6 border-t border-gray-200 pt-4">
        <h3 className="text-sm font-semibold uppercase text-gray-700 mb-3">
          Filter by Price
        </h3>

        <input
          type="range"
          min={minPrice}
          max={maxPrice}
          value={priceRange[0]}
          onChange={(e) =>
            onPriceChange([+e.target.value, priceRange[1]])
          }
          className="w-full accent-black"
        />

        <input
          type="range"
          min={minPrice}
          max={maxPrice}
          value={priceRange[1]}
          onChange={(e) =>
            onPriceChange([priceRange[0], +e.target.value])
          }
          className="w-full accent-black mt-2"
        />

        <p className="text-sm mt-2 text-gray-500">
          ${priceRange[0]} – ${priceRange[1]}
        </p>
      </div>

      {/* Brand Filter */}
      <div className="border-t border-gray-200 pt-4">
        <h3 className="text-sm font-semibold uppercase text-gray-700 mb-3">
          Filter by Brand
        </h3>

        <div className="space-y-2">
          {brands.map((brand) => (
            <label
              key={brand.id}
              className="flex items-center gap-2 text-sm text-gray-600 cursor-pointer"
            >
              <input
                type="checkbox"
                checked={selectedBrands.includes(brand.id)}
                onChange={() => toggleBrand(brand.id)}
                className="accent-black"
              />
              <span className="hover:text-gray-800">
                {brand.name}
              </span>
            </label>
          ))}
        </div>
      </div>
    </aside>
  );
};

export default Sidebar;
