'use client';

import { useEffect, useState, useMemo } from "react";
import { X } from "lucide-react";
import toast from "react-hot-toast";
import Link from "next/link";

const IMAGE_BASE = process.env.NEXT_PUBLIC_IMAGE_BASE + '/product';

export default function CartPage() {
    const [cartItems, setCartItems] = useState([]);
    const [loading, setLoading] = useState(true);

    const [shipping, setShipping] = useState("local");
    const [coupon, setCoupon] = useState("");
    const [discount, setDiscount] = useState(0);

    const getCookie = (name) => {
        const value = `; ${document.cookie}`;
        const parts = value.split(`; ${name}=`);
        if (parts.length === 2) return parts.pop().split(';').shift();
    };

    /* -----------------------------
       FETCH CART
    ------------------------------*/
    useEffect(() => {
        const cartId = getCookie('cart_id');

        fetch(
            `${process.env.NEXT_PUBLIC_API_URL}/my-cart?cart_id=${cartId}`,
            {
                method: 'GET',
                credentials: 'include',
            }
        )
            .then(res => res.json())
            .then(res => {
                if (res.status === 'success') {
                    const mapped = res.data.map(item => ({
                        id: item.id,
                        name: item.product.name,
                        price: Number(item.product.base_price),
                        quantity: item.quantity,
                        img: `${IMAGE_BASE}/${item.product.main_image}`,
                    }));
                    setCartItems(mapped);
                }
            })
            .finally(() => setLoading(false));
    }, []);

    /* -----------------------------
       CALCULATIONS
    ------------------------------*/
    const subtotal = useMemo(
        () => cartItems.reduce((sum, i) => sum + i.price * i.quantity, 0),
        [cartItems]
    );

    const shippingCost = shipping === "flat" ? 5 : 0;

    const total = useMemo(
        () => subtotal - discount + shippingCost,
        [subtotal, discount, shippingCost]
    );

    /* -----------------------------
       REMOVE ITEM (API)
    ------------------------------*/
    const removeItem = async (id) => {
        const toastId = toast.loading("Removing item...");

        try {
            const res = await fetch(
                `${process.env.NEXT_PUBLIC_API_URL}/remove_cart_item/${id}`,
                {
                    method: 'GET',
                    credentials: 'include',
                }
            );

            const data = await res.json();

            if (data.success) {
                setCartItems(prev => prev.filter(i => i.id !== id));
                toast.success(data.success, { id: toastId });
            } else {
                toast.error(data.error || "Item not found", { id: toastId });
            }
        } catch {
            toast.error("Something went wrong", { id: toastId });
        }
    };

    /* -----------------------------
       UPDATE QUANTITY (API)
    ------------------------------*/
    const updateQuantity = async (id, newQty) => {
        if (newQty < 1) return;

        const toastId = toast.loading("Updating quantity...");

        try {
            const res = await fetch(
                `${process.env.NEXT_PUBLIC_API_URL}/update_cart_item/${id}?quantity=${newQty}`,
                {
                    method: 'GET',
                    credentials: 'include',
                }
            );

            const data = await res.json();

            if (data.success) {
                setCartItems(prev =>
                    prev.map(i =>
                        i.id === id ? { ...i, quantity: newQty } : i
                    )
                );
                toast.success(data.success, { id: toastId });
            } else {
                toast.error(data.error || "Update failed", { id: toastId });
            }
        } catch {
            toast.error("Something went wrong", { id: toastId });
        }
    };

    const applyCoupon = () => {
        if (coupon.toLowerCase() === "save10") {
            setDiscount(10);
        } else {
            setDiscount(0);
            toast.error("Invalid coupon code");
        }
    };

    if (loading) {
        return <p className="p-10">Loading cart...</p>;
    }

    return (
        <main className="max-w-7xl mx-auto">
            <div className="container mx-auto px-4 py-10 grid md:grid-cols-3 gap-8">

                {/* ================= CART ITEMS ================= */}
                <div className="md:col-span-2">
                    {cartItems.length === 0 && (
                        <p className="text-gray-500">Your cart is empty.</p>
                    )}

                    {cartItems.map(item => (
                        <div
                            key={item.id}
                            className="flex items-center border-b border-gray-200 py-4 relative"
                        >
                            {/* Image */}
                            <div className="relative w-24 h-24 flex-shrink-0">
                                <img
                                    src={item.img}
                                    alt={item.name}
                                    className="w-full h-full object-cover rounded"
                                />
                                <button
                                    onClick={() => removeItem(item.id)}
                                    className="absolute -top-2 -left-2 bg-white border border-gray-300 rounded-full p-1 text-gray-500 hover:text-red-500 shadow-sm"
                                >
                                    <X size={14} />
                                </button>
                            </div>

                            {/* Info */}
                            <div className="flex-1 ml-4">
                                <h3 className="text-gray-800 font-medium">
                                    {item.name}
                                </h3>
                                <p className="text-gray-500">
                                    ৳ {item.price.toFixed(2)}
                                </p>
                            </div>

                            {/* Quantity */}
                            <div className="flex items-center">
                                <button
                                    onClick={() => updateQuantity(item.id, item.quantity - 1)}
                                    className="px-4 py-1 border border-gray-300"
                                >
                                    −
                                </button>
                                <span className="px-4 py-1 border border-gray-300">
                                    {item.quantity}
                                </span>
                                <button
                                    onClick={() => updateQuantity(item.id, item.quantity + 1)}
                                    className="px-4 py-1 border border-gray-300"
                                >
                                    +
                                </button>
                            </div>

                            {/* Line Total */}
                            <div className="w-24 text-right font-semibold">
                                ৳ {(item.price * item.quantity).toFixed(2)}
                            </div>
                        </div>
                    ))}

                    {/* Coupon */}
                    <div className="flex flex-col sm:flex-row justify-between items-center mt-6 gap-3">
                        <div className="flex w-full sm:w-auto">
                            <input
                                type="text"
                                placeholder="Coupon Code"
                                value={coupon}
                                onChange={e => setCoupon(e.target.value)}
                                className="border border-gray-300 px-4 py-2 w-full sm:w-64 rounded-l"
                            />
                            <button
                                onClick={applyCoupon}
                                className="bg-gray-800 text-white px-4 py-2 rounded-r"
                            >
                                APPLY COUPON
                            </button>
                        </div>
                    </div>
                </div>

                {/* ================= TOTALS ================= */}
                <div className="border border-gray-200 p-6 rounded-md h-fit">
                    <h2 className="text-lg font-semibold mb-4">
                        CART TOTALS
                    </h2>

                    <div className="flex justify-between py-2 border-b">
                        <span>Subtotal</span>
                        <span>৳ {subtotal.toFixed(2)}</span>
                    </div>

                    <div className="py-3 border-b">
                        <h3 className="font-medium mb-2">Shipping</h3>

                        <div className="flex items-center mb-1">
                            <input
                                type="radio"
                                checked={shipping === "local"}
                                onChange={() => setShipping("local")}
                                className="mr-2"
                            />
                            <label>Local pickup</label>
                        </div>

                        <div className="flex items-center">
                            <input
                                type="radio"
                                checked={shipping === "flat"}
                                onChange={() => setShipping("flat")}
                                className="mr-2"
                            />
                            <label>Flat rate (৳5.00)</label>
                        </div>
                    </div>

                    <div className="flex justify-between py-3 text-lg font-semibold">
                        <span>Total</span>
                        <span>৳ {total.toFixed(2)}</span>
                    </div>

                    <Link href={`/checkout`} className="block">
                        <button className="bg-gray-800 text-white w-full py-3 rounded mt-3 cursor-pointer">
                            PROCEED TO CHECKOUT →
                        </button>
                    </Link>
                </div>
            </div>
        </main>
    );
}
