"use client";

import { useEffect, useState } from "react";
import { Home, ChevronRight, Grid3X3, List } from "lucide-react";
import Sidebar from "../components/category/Sidebar";
import ProductCard from "../components/category/ProductCard";

export default function CategoryClient({ categoryId, categories }) {
  const [viewType, setViewType] = useState("grid");
  const [sortBy, setSortBy] = useState("default");

  const [showCount, setShowCount] = useState(12);
  const [currentPage, setCurrentPage] = useState(1);

  const [selectedBrands, setSelectedBrands] = useState([]);
  const [priceRange, setPriceRange] = useState([0, 100000]);

  const [products, setProducts] = useState([]);
  const [brands, setBrands] = useState([]);
  const [category, setCategory] = useState({});
  const [pagination, setPagination] = useState({});

  const [loading, setLoading] = useState(true);

  /* -------------------------------
     BUILD QUERY STRING
  --------------------------------*/
  const buildQuery = () => {
    const params = new URLSearchParams();

    selectedBrands.forEach((id) => {
      params.append("brand[]", id);
    });

    params.append("min", priceRange[0]);
    params.append("max", priceRange[1]);
    params.append("perpage", showCount);
    params.append("page", currentPage);

    if (sortBy !== "default") {
      params.append("sort", sortBy);
    }

    return params.toString();
  };

  /* -------------------------------
     FETCH PRODUCTS
  --------------------------------*/
  useEffect(() => {
    if (!categoryId) return;

    setLoading(true);

    const query = buildQuery();

    fetch(
      `${process.env.NEXT_PUBLIC_API_URL}/categories/${categoryId}?${query}`
    )
      .then((res) => res.json())
      .then((res) => {
        const apiProducts = res?.data?.products?.data || [];

        console.log(res)

        setBrands(res?.data?.brands || []);
        setCategory(res?.data?.category || {});
        setPagination(res?.data?.products || {});

        const formattedProducts = apiProducts.map((item) => {
          const price = Number(item.base_price);

          return {
            id: item.id,
            name: item.name,
            slug: item.slug,
            category: item.categories?.[0]?.name ?? "",
            originalPrice: price,
            salePrice: price,
            rating: item.reviews?.length ?? 0,
            image: `${process.env.NEXT_PUBLIC_IMAGE_BASE}/product/${item.main_image}`,
            badge: item.is_featured ? "FEATURED" : null,
            buttonType: item.has_variants ? "options" : "cart",
          };
        });

        setProducts(formattedProducts);
      })
      .catch((err) => {
        console.error("Product fetch error:", err);
        setProducts([]);
      })
      .finally(() => setLoading(false));
  }, [
    categoryId,
    selectedBrands,
    priceRange,
    showCount,
    currentPage,
    sortBy,
  ]);

  return (
    <div className="max-w-7xl mx-auto px-4 py-6">
      {/* Breadcrumb */}
      <nav className="flex items-center gap-2 text-sm text-gray-400 mb-6">
        <Home size={14} />
        <ChevronRight size={14} />
        <span>Home</span>
        <ChevronRight size={14} />
        <span>Category</span>
        <ChevronRight size={14} />
        <span className="text-gray-600">{category?.name}</span>
      </nav>

      <div className="flex flex-col lg:flex-row gap-8">
        {/* Sidebar */}
        <Sidebar
          activeCategoryId={categoryId}
          categories={categories}
          brands={brands}
          selectedBrands={selectedBrands}
          priceRange={priceRange}
          onBrandChange={setSelectedBrands}
          onPriceChange={setPriceRange}
        />

        {/* Main Content */}
        <main className="flex-1">
          {/* Toolbar */}
          <div className="flex justify-between mb-6">
            <div className="flex items-center gap-2">
              <span className="text-sm text-gray-400">Sort By:</span>
              <select
                value={sortBy}
                onChange={(e) => setSortBy(e.target.value)}
                className="border px-3 py-1.5 text-sm"
              >
                <option value="default">Default</option>
                <option value="price-low">Price: Low → High</option>
                <option value="price-high">Price: High → Low</option>
              </select>
            </div>

            <div className="flex gap-2">
              <button onClick={() => setViewType("grid")}>
                <Grid3X3 size={18} />
              </button>
              <button onClick={() => setViewType("list")}>
                <List size={18} />
              </button>
            </div>
          </div>

          {/* Products */}
          <div
            className={`grid gap-6 ${
              viewType === "grid"
                ? "grid-cols-2 md:grid-cols-3"
                : "grid-cols-1"
            }`}
          >
            {loading && <p className="text-gray-400">Loading products...</p>}

            {!loading && products.length === 0 && (
              <p className="text-gray-400">No products found</p>
            )}

            {products.map((p) => (
              <ProductCard key={p.id} product={p} />
            ))}
          </div>

          {/* Pagination */}
          {pagination?.last_page > 1 && (
            <div className="flex items-center justify-between mt-8">
              <select
                value={showCount}
                onChange={(e) => {
                  setShowCount(Number(e.target.value));
                  setCurrentPage(1);
                }}
                className="border px-3 py-1.5 text-sm"
              >
                <option value={12}>12</option>
                <option value={24}>24</option>
                <option value={36}>36</option>
              </select>

              <div className="flex gap-2">
                {[...Array(pagination.last_page)].map((_, i) => (
                  <button
                    key={i}
                    onClick={() => setCurrentPage(i + 1)}
                    className={`w-8 h-8 border text-sm ${
                      currentPage === i + 1
                        ? "bg-gray-200 text-gray-700"
                        : "text-gray-400"
                    }`}
                  >
                    {i + 1}
                  </button>
                ))}
              </div>
            </div>
          )}
        </main>
      </div>
    </div>
  );
}
